'use client';

import React, { useState } from 'react';
import { useParams, useRouter } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    Grid,
    Card,
    CardContent,
    Avatar,
    Rating,
    Button,
    Chip,
    Tabs,
    Tab,
    Divider,
    List,
    ListItem,
    Paper,
    LinearProgress,
} from '@mui/material';
import {
    Star,
    LocationOn,
    Language,
    AccessTime,
    CheckCircle,
    WorkOutline,
} from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import ServiceCard from '@/components/ServiceCard';
import { freelancers, services, reviews, portfolioItems } from '@/data/mockData';

export default function FreelancerProfilePage() {
    const params = useParams();
    const router = useRouter();
    const freelancerId = parseInt(params.id);

    const freelancer = freelancers.find((f) => f.id === freelancerId);
    const freelancerServices = services.filter((s) => s.provider.id === freelancerId);
    const freelancerReviews = reviews.filter((r) => r.freelancerId === freelancerId);
    const portfolio = portfolioItems.filter((p) => p.freelancerId === freelancerId);

    const [selectedTab, setSelectedTab] = useState(0);

    if (!freelancer) {
        return (
            <>
                <Header />
                <Container sx={{ py: 8, textAlign: 'center' }}>
                    <Typography variant="h4">Freelancer not found</Typography>
                    <Button variant="contained" onClick={() => router.push('/freelancers')} sx={{ mt: 2 }}>
                        Browse Freelancers
                    </Button>
                </Container>
                <Footer />
            </>
        );
    }

    // Calculate rating distribution
    const ratingDistribution = [5, 4, 3, 2, 1].map((star) => ({
        star,
        count: freelancerReviews.filter((r) => r.rating === star).length,
        percentage: freelancerReviews.length > 0
            ? (freelancerReviews.filter((r) => r.rating === star).length / freelancerReviews.length) * 100
            : 0,
    }));

    return (
        <>
            <Header />

            {/* Profile Header */}
            <Box
                sx={{
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    color: 'white',
                    py: 6,
                }}
            >
                <Container maxWidth="lg">
                    <Grid container spacing={3} alignItems="center">
                        <Grid item>
                            <Avatar
                                src={freelancer.avatar}
                                alt={freelancer.name}
                                sx={{
                                    width: 120,
                                    height: 120,
                                    border: '4px solid white',
                                }}
                            />
                        </Grid>
                        <Grid item xs>
                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                                <Typography variant="h3" fontWeight={700}>
                                    {freelancer.name}
                                </Typography>
                                {freelancer.isPremium && (
                                    <Chip
                                        label="Premium"
                                        size="small"
                                        sx={{
                                            bgcolor: '#f59e0b',
                                            color: 'white',
                                            fontWeight: 600,
                                        }}
                                    />
                                )}
                            </Box>
                            <Typography variant="h6" sx={{ opacity: 0.9, mb: 2 }}>
                                {freelancer.title}
                            </Typography>
                            <Box sx={{ display: 'flex', gap: 3, flexWrap: 'wrap' }}>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                    <Star sx={{ color: '#f59e0b' }} />
                                    <Typography variant="h6" fontWeight={600}>
                                        {freelancer.rating}
                                    </Typography>
                                    <Typography variant="body2" sx={{ opacity: 0.9 }}>
                                        ({freelancer.reviews} reviews)
                                    </Typography>
                                </Box>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                    <LocationOn />
                                    <Typography variant="body1">{freelancer.location}</Typography>
                                </Box>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                    <WorkOutline />
                                    <Typography variant="body1">{freelancer.completedProjects} projects</Typography>
                                </Box>
                            </Box>
                        </Grid>
                        <Grid item>
                            <Paper sx={{ p: 3, textAlign: 'center' }}>
                                <Typography variant="h4" fontWeight={700} color="primary.main">
                                    ${freelancer.hourlyRate}
                                </Typography>
                                <Typography variant="body2" color="text.secondary">
                                    per hour
                                </Typography>
                                <Button variant="contained" size="large" fullWidth sx={{ mt: 2 }}>
                                    Contact Me
                                </Button>
                            </Paper>
                        </Grid>
                    </Grid>
                </Container>
            </Box>

            <Container maxWidth="lg" sx={{ mt: 4, mb: 8 }}>
                <Grid container spacing={4}>
                    {/* Main Content */}
                    <Grid item xs={12} md={8}>
                        <Card sx={{ mb: 3 }}>
                            <Tabs value={selectedTab} onChange={(e, v) => setSelectedTab(v)}>
                                <Tab label="About" />
                                <Tab label={`Portfolio (${portfolio.length})`} />
                                <Tab label={`Services (${freelancerServices.length})`} />
                                <Tab label={`Reviews (${freelancerReviews.length})`} />
                            </Tabs>
                            <Divider />

                            <CardContent>
                                {/* About Tab */}
                                {selectedTab === 0 && (
                                    <Box>
                                        <Typography variant="h6" fontWeight={600} gutterBottom>
                                            About Me
                                        </Typography>
                                        <Typography variant="body1" paragraph sx={{ lineHeight: 1.8 }}>
                                            {freelancer.bio}
                                        </Typography>

                                        <Typography variant="h6" fontWeight={600} gutterBottom sx={{ mt: 3 }}>
                                            Skills
                                        </Typography>
                                        <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                                            {freelancer.skills.map((skill) => (
                                                <Chip key={skill} label={skill} color="primary" variant="outlined" />
                                            ))}
                                        </Box>
                                    </Box>
                                )}

                                {/* Portfolio Tab */}
                                {selectedTab === 1 && (
                                    <Box>
                                        {portfolio.length === 0 ? (
                                            <Typography color="text.secondary">No portfolio items yet</Typography>
                                        ) : (
                                            <Grid container spacing={3}>
                                                {portfolio.map((item) => (
                                                    <Grid item xs={12} sm={6} key={item.id}>
                                                        <Card>
                                                            <Box
                                                                component="img"
                                                                src={item.image}
                                                                alt={item.title}
                                                                sx={{
                                                                    width: '100%',
                                                                    height: 200,
                                                                    objectFit: 'cover',
                                                                }}
                                                            />
                                                            <CardContent>
                                                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                                                    {item.title}
                                                                </Typography>
                                                                <Typography variant="body2" color="text.secondary" paragraph>
                                                                    {item.description}
                                                                </Typography>
                                                                <Box sx={{ display: 'flex', gap: 0.5, flexWrap: 'wrap' }}>
                                                                    {item.tags.map((tag) => (
                                                                        <Chip key={tag} label={tag} size="small" />
                                                                    ))}
                                                                </Box>
                                                            </CardContent>
                                                        </Card>
                                                    </Grid>
                                                ))}
                                            </Grid>
                                        )}
                                    </Box>
                                )}

                                {/* Services Tab */}
                                {selectedTab === 2 && (
                                    <Box>
                                        {freelancerServices.length === 0 ? (
                                            <Typography color="text.secondary">No services available</Typography>
                                        ) : (
                                            <Grid container spacing={3}>
                                                {freelancerServices.map((service) => (
                                                    <Grid item xs={12} sm={6} key={service.id}>
                                                        <ServiceCard {...service} />
                                                    </Grid>
                                                ))}
                                            </Grid>
                                        )}
                                    </Box>
                                )}

                                {/* Reviews Tab */}
                                {selectedTab === 3 && (
                                    <Box>
                                        {freelancerReviews.length === 0 ? (
                                            <Typography color="text.secondary">No reviews yet</Typography>
                                        ) : (
                                            <List>
                                                {freelancerReviews.map((review) => (
                                                    <React.Fragment key={review.id}>
                                                        <ListItem alignItems="flex-start" sx={{ px: 0 }}>
                                                            <Box sx={{ width: '100%' }}>
                                                                <Box sx={{ display: 'flex', gap: 2, mb: 1 }}>
                                                                    <Avatar src={review.clientAvatar} alt={review.clientName} />
                                                                    <Box sx={{ flex: 1 }}>
                                                                        <Typography variant="subtitle2" fontWeight={600}>
                                                                            {review.clientName}
                                                                        </Typography>
                                                                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                                                            <Rating value={review.rating} size="small" readOnly />
                                                                            <Typography variant="caption" color="text.secondary">
                                                                                {review.date}
                                                                            </Typography>
                                                                        </Box>
                                                                    </Box>
                                                                </Box>
                                                                <Typography variant="body2" sx={{ ml: 7 }}>
                                                                    {review.comment}
                                                                </Typography>
                                                            </Box>
                                                        </ListItem>
                                                        <Divider sx={{ my: 2 }} />
                                                    </React.Fragment>
                                                ))}
                                            </List>
                                        )}
                                    </Box>
                                )}
                            </CardContent>
                        </Card>
                    </Grid>

                    {/* Sidebar */}
                    <Grid item xs={12} md={4}>
                        {/* Stats Card */}
                        <Card sx={{ mb: 3 }}>
                            <CardContent>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Statistics
                                </Typography>
                                <Divider sx={{ my: 2 }} />

                                <Box sx={{ mb: 2 }}>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Response Time
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.responseTime}
                                        </Typography>
                                    </Box>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Completed Projects
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.completedProjects}
                                        </Typography>
                                    </Box>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography variant="body2" color="text.secondary">
                                            Member Since
                                        </Typography>
                                        <Typography variant="body2" fontWeight={600}>
                                            {freelancer.memberSince}
                                        </Typography>
                                    </Box>
                                </Box>
                            </CardContent>
                        </Card>

                        {/* Languages Card */}
                        <Card sx={{ mb: 3 }}>
                            <CardContent>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 2 }}>
                                    <Language />
                                    <Typography variant="h6" fontWeight={600}>
                                        Languages
                                    </Typography>
                                </Box>
                                <Box sx={{ display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                                    {freelancer.languages.map((lang) => (
                                        <Chip key={lang} label={lang} />
                                    ))}
                                </Box>
                            </CardContent>
                        </Card>

                        {/* Rating Breakdown */}
                        {freelancerReviews.length > 0 && (
                            <Card>
                                <CardContent>
                                    <Typography variant="h6" fontWeight={600} gutterBottom>
                                        Rating Breakdown
                                    </Typography>
                                    <Box sx={{ textAlign: 'center', mb: 3 }}>
                                        <Typography variant="h2" fontWeight={700} color="primary.main">
                                            {freelancer.rating}
                                        </Typography>
                                        <Rating value={freelancer.rating} precision={0.1} readOnly />
                                        <Typography variant="body2" color="text.secondary">
                                            Based on {freelancer.reviews} reviews
                                        </Typography>
                                    </Box>

                                    {ratingDistribution.map(({ star, count, percentage }) => (
                                        <Box key={star} sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 1 }}>
                                            <Typography variant="body2" sx={{ minWidth: 60 }}>
                                                {star} stars
                                            </Typography>
                                            <LinearProgress
                                                variant="determinate"
                                                value={percentage}
                                                sx={{ flex: 1, height: 8, borderRadius: 4 }}
                                            />
                                            <Typography variant="body2" sx={{ minWidth: 30 }}>
                                                {count}
                                            </Typography>
                                        </Box>
                                    ))}
                                </CardContent>
                            </Card>
                        )}
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}
